//	TorusGames2DIntro.c
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#include "TorusGames-Common.h"
#include "GeometryGamesUtilities-Common.h"
#include <math.h>


#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_INTRO		2

//	Size of each flounder or seashell.
//	(The hit testing is still a bit of a hack.)
#define FLOUNDER_SIZE		0.50
#define CRAB_SIZE			0.30
#define SHELL_SIZE_A		0.20
#define SHELL_SIZE_B		0.20
#define OBJECT_HIT_RADIUS	0.20


//	Public functions with private names
static void	IntroReset(ModelData *md);
static bool	IntroDragBegin(ModelData *md, bool aRightClick);
static void	IntroDragObject(ModelData *md, double aHandLocalDeltaH, double aHandLocalDeltaV);

//	Private functions
//	<no private functions>


void Intro2DSetUp(ModelData *md)
{
	//	Initialize function pointers.
	md->itsGameShutDown					= NULL;
	md->itsGameReset					= &IntroReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= NULL;
	md->itsGame2DDragBegin				= &IntroDragBegin;
	md->itsGame2DDragObject				= &IntroDragObject;
	md->itsGame2DDragEnd				= NULL;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= NULL;
	md->itsGameSimulationUpdate			= NULL;
	md->itsGameRefreshMessage			= NULL;

	//	Reset the scene.
	IntroReset(md);
}


static void IntroReset(ModelData *md)
{
	//	Place the flounder in the center.

	md->itsGameOf.Intro2D.itsSpritePlacement[0].itsH		= 0.0;
	md->itsGameOf.Intro2D.itsSpritePlacement[0].itsV		= 0.0;
	md->itsGameOf.Intro2D.itsSpritePlacement[0].itsFlip		= false;
	md->itsGameOf.Intro2D.itsSpritePlacement[0].itsAngle	= 0.0;
	md->itsGameOf.Intro2D.itsSpritePlacement[0].itsSizeH	= FLOUNDER_SIZE;
	md->itsGameOf.Intro2D.itsSpritePlacement[0].itsSizeV	= FLOUNDER_SIZE;

	//	Place the crab and the shells arbitrarily, but avoiding the flounder.

	md->itsGameOf.Intro2D.itsSpritePlacement[1].itsH		= -0.45;
	md->itsGameOf.Intro2D.itsSpritePlacement[1].itsV		= -0.30;
	md->itsGameOf.Intro2D.itsSpritePlacement[1].itsFlip		= RandomBoolean();
	md->itsGameOf.Intro2D.itsSpritePlacement[1].itsAngle	= TWOPI * RandomFloat();
	md->itsGameOf.Intro2D.itsSpritePlacement[1].itsSizeH	= CRAB_SIZE;
	md->itsGameOf.Intro2D.itsSpritePlacement[1].itsSizeV	= CRAB_SIZE;

	md->itsGameOf.Intro2D.itsSpritePlacement[2].itsH		= -0.3;
	md->itsGameOf.Intro2D.itsSpritePlacement[2].itsV		= +0.4;
	md->itsGameOf.Intro2D.itsSpritePlacement[2].itsFlip		= RandomBoolean();
	md->itsGameOf.Intro2D.itsSpritePlacement[2].itsAngle	= TWOPI * RandomFloat();
	md->itsGameOf.Intro2D.itsSpritePlacement[2].itsSizeH	= SHELL_SIZE_A;
	md->itsGameOf.Intro2D.itsSpritePlacement[2].itsSizeV	= SHELL_SIZE_A;

	md->itsGameOf.Intro2D.itsSpritePlacement[3].itsH		= +0.3;
	md->itsGameOf.Intro2D.itsSpritePlacement[3].itsV		= +0.3;
	md->itsGameOf.Intro2D.itsSpritePlacement[3].itsFlip		= RandomBoolean();
	md->itsGameOf.Intro2D.itsSpritePlacement[3].itsAngle	= TWOPI * RandomFloat();
	md->itsGameOf.Intro2D.itsSpritePlacement[3].itsSizeH	= SHELL_SIZE_B;
	md->itsGameOf.Intro2D.itsSpritePlacement[3].itsSizeV	= SHELL_SIZE_B;

	//	Abort any pending simulation.
	SimulationEnd(md);	//	unused in Intro2D game

	//	Just for good form...
	md->itsGameOf.Intro2D.itsDragSprite	= 0;		//	ignored until drag begins
	md->itsGameIsOver					= false;	//	unused in Intro2D game
}


static bool IntroDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	unsigned int	i,
					j,
					k;

	UNUSED_PARAMETER(aRightClick);
	
	for (i = 0; i < NUM_2D_INTRO_SPRITES; i++)
	{
		md->its2DDragIsReversed = (	md->its2DHandPlacement.itsFlip
								 != md->itsGameOf.Intro2D.itsSpritePlacement[i].itsFlip);

		if (Shortest2DDistance(
					md->its2DHandPlacement.itsH,
					md->its2DHandPlacement.itsV,
					md->itsGameOf.Intro2D.itsSpritePlacement[i].itsH,
					md->itsGameOf.Intro2D.itsSpritePlacement[i].itsV,
					md->itsTopology,
					&md->its2DDragIsReversed)
				< OBJECT_HIT_RADIUS)
		{
			//	Record which sprite was hit.
			md->itsGameOf.Intro2D.itsDragSprite = i;
			
			//	Clear the history.
			for (j = 0; j < DRAG_HISTORY_DEPTH; j++)
				for (k = 0; k < 2; k++)
					md->its2DDragHistory[j][k] = 0.0;

			//	Report a hit.
			return true;
		}
	}

	return false;
}


static void IntroDragObject(
	ModelData	*md,
	double		aHandLocalDeltaH,
	double		aHandLocalDeltaV)
{
	double			theSpriteLocalDeltaH,
					theSpriteLocalDeltaV,
					theHandAmbientDeltaH,
					theHandAmbientDeltaV,
					theSpriteAmbientDeltaH,
					theSpriteAmbientDeltaV,
					theNewPositionH,
					theNewPositionV,
					theTotalSpriteLocalDeltaH,
					theTotalSpriteLocalDeltaV;
	unsigned int	i,
					j;

	if (md->itsGameOf.Intro2D.itsDragSprite >= NUM_2D_INTRO_SPRITES)	//	should never occur
		return;

	theSpriteLocalDeltaH	= md->its2DDragIsReversed
								? -aHandLocalDeltaH : aHandLocalDeltaH;
	theSpriteLocalDeltaV	= aHandLocalDeltaV;

	theHandAmbientDeltaH	= md->its2DHandPlacement.itsFlip
								? -aHandLocalDeltaH : aHandLocalDeltaH;
	theHandAmbientDeltaV	= aHandLocalDeltaV;

	theSpriteAmbientDeltaH	= md->itsGameOf.Intro2D.itsSpritePlacement[md->itsGameOf.Intro2D.itsDragSprite].itsFlip
								? -theSpriteLocalDeltaH : theSpriteLocalDeltaH;
	theSpriteAmbientDeltaV	= theSpriteLocalDeltaV;

	theNewPositionH = md->itsGameOf.Intro2D.itsSpritePlacement[md->itsGameOf.Intro2D.itsDragSprite].itsH + theSpriteAmbientDeltaH;
	theNewPositionV = md->itsGameOf.Intro2D.itsSpritePlacement[md->itsGameOf.Intro2D.itsDragSprite].itsV + theSpriteAmbientDeltaV;

	//	Move the sprite.
	md->itsGameOf.Intro2D.itsSpritePlacement[md->itsGameOf.Intro2D.itsDragSprite].itsH = theNewPositionH;
	md->itsGameOf.Intro2D.itsSpritePlacement[md->itsGameOf.Intro2D.itsDragSprite].itsV = theNewPositionV;
	Normalize2DPlacement(&md->itsGameOf.Intro2D.itsSpritePlacement[md->itsGameOf.Intro2D.itsDragSprite], md->itsTopology);
	//	Note:  The sprite's angle will get set below.

	//	Move the hand.
	md->its2DHandPlacement.itsH += theHandAmbientDeltaH;
	md->its2DHandPlacement.itsV += theHandAmbientDeltaV;
	Normalize2DPlacement(&md->its2DHandPlacement, md->itsTopology);

	//	Only the flounder and the crab change direction when dragged.
	if (md->itsGameOf.Intro2D.itsDragSprite == 0	//	flounder
	 || md->itsGameOf.Intro2D.itsDragSprite == 1)	//	crab
	{
		//	Update the drag history.
		for (i = DRAG_HISTORY_DEPTH - 1; i-- > 0; )
			for (j = 0; j < 2; j++)
				md->its2DDragHistory[i+1][j] = md->its2DDragHistory[i][j];
		md->its2DDragHistory[0][0] = theSpriteLocalDeltaH;
		md->its2DDragHistory[0][1] = theSpriteLocalDeltaV;

		//	Adjust the sprite's heading.
		//	To avoid processor-speed dependencies, sum the sprite's local motions
		//	over a fixed distance rather than a fixed number of frames.
		theTotalSpriteLocalDeltaH = 0.0;
		theTotalSpriteLocalDeltaV = 0.0;
		for (i = 0; i < DRAG_HISTORY_DEPTH; i++)
		{
			theTotalSpriteLocalDeltaH += md->its2DDragHistory[i][0];
			theTotalSpriteLocalDeltaV += md->its2DDragHistory[i][1];

			if (fabs(theTotalSpriteLocalDeltaH) > 0.1
			 || fabs(theTotalSpriteLocalDeltaV) > 0.1)
				break;
		}
		if (theTotalSpriteLocalDeltaH != 0.0
		 || theTotalSpriteLocalDeltaV != 0.0)
			md->itsGameOf.Intro2D.itsSpritePlacement[md->itsGameOf.Intro2D.itsDragSprite].itsAngle = atan2(theTotalSpriteLocalDeltaV, theTotalSpriteLocalDeltaH);
	}
}


unsigned int GetNum2DIntroBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_INTRO;
}

void Get2DIntroKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 1.00;
	someKleinAxisColors[0][1] = 0.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.00;
	someKleinAxisColors[1][1] = 0.00;
	someKleinAxisColors[1][2] = 1.00;
	someKleinAxisColors[1][3] = 1.00;
}


unsigned int GetNum2DIntroSprites(void)
{
	return NUM_2D_INTRO_SPRITES;	//	flounder, crab, clam, scallop
}

void Get2DIntroSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	unsigned int	i;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DIntro,
		"Game2DIntro must be active");

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DIntroSprites(),
		"Internal error:  wrong buffer size");

	for (i = 0; i < NUM_2D_INTRO_SPRITES; i++)
		aPlacementBuffer[i] = md->itsGameOf.Intro2D.itsSpritePlacement[i];
}

